<?php
@ini_set('display_errors','1'); @error_reporting(E_ALL);

require_once __DIR__ . '/../includes/initialize.php';
global $con;
if (!$con) { die('DB connection not initialized'); }

require_once __DIR__ . '/../includes/auth.php';
require_login();

/* ---------------------------
   VIEW-ONLY ACL GUARD (UPDATED MATCHING)
   - Uses jos_admin_menus.menu_link for menu matching (normalized path)
   - Allows ?menu_id= override for testing
   - If no menu mapping is found -> DENY (403)
   - Matching improved: exact, basename, directory-prefix, index.php directory fallback
   - If jos_admin_menus.can_view = 1 for the matched menu -> allow
   - Otherwise attempts role/user permission checks and requires can_view = 1
   --------------------------- */
(function() use ($con) {
  // Local helpers
  $get_int = function($k, $d=0){ return isset($_GET[$k]) ? (int)$_GET[$k] : $d; };

  // Allow overriding menu check with ?menu_id= for testing
  $override_menu_id = $get_int('menu_id', 0);

  // Determine normalized request path (no querystring, collapsed slashes, trim leading slash)
  $req_path = '';
  if (!empty($_SERVER['REQUEST_URI'])) {
    $req_path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $req_path = preg_replace('#/+#', '/', (string)$req_path);
    $req_path = ltrim($req_path, '/');
  } elseif (!empty($_SERVER['SCRIPT_NAME'])) {
    $req_path = ltrim($_SERVER['SCRIPT_NAME'], '/');
  }
  $req_path = (string)$req_path;

  $matched_menu_id = 0;
  if ($override_menu_id > 0) {
    $matched_menu_id = $override_menu_id;
  } else {
    // Try to find a match in jos_admin_menus.menu_link with improved logic
    $sql = "SELECT id, menu_link FROM jos_admin_menus";
    if ($rs = $con->query($sql)) {
      while ($m = $rs->fetch_assoc()) {
        $mlink = (string)$m['menu_link'];
        $mlink_path = parse_url($mlink, PHP_URL_PATH);
        $mlink_path = preg_replace('#/+#', '/', (string)$mlink_path);
        $mlink_path = ltrim($mlink_path, '/');

        // normalize directory forms
        $mlink_dir = rtrim(dirname($mlink_path), '/');
        $req_dir  = rtrim(dirname($req_path), '/');

        // Exact normalized match
        if ($mlink_path === $req_path) {
          $matched_menu_id = (int)$m['id'];
          break;
        }

        // Basename match (e.g., both end with index.php) - helpful when menu links point to index pages
        if ($req_path !== '' && basename($mlink_path) === basename($req_path)) {
          $matched_menu_id = (int)$m['id'];
          break;
        }

        // If menu link is a directory (ends with '/'), allow prefix match: request path starts with that directory
        // e.g., menu_link '/adminconsole/operations/' should match '/adminconsole/operations/standard_jobs_report.php'
        if ($mlink_path !== '' && (substr($mlink_path, -1) === '/' || basename($mlink_path) === 'index.php')) {
          $prefix = rtrim($mlink_path, '/');
          if ($prefix !== '' && stripos($req_path, $prefix . '/') === 0) {
            $matched_menu_id = (int)$m['id'];
            break;
          }
        }

        // If request is inside the menu directory (compare directories)
        if ($mlink_dir !== '' && $req_dir !== '' && (stripos($req_dir, $mlink_dir) === 0 || stripos($mlink_dir, $req_dir) === 0)) {
          $matched_menu_id = (int)$m['id'];
          break;
        }

        // Fallback: if menu link is contained in request path or vice-versa (loose match)
        if ($mlink_path !== '' && $req_path !== '' && (stripos($req_path, $mlink_path) !== false || stripos($mlink_path, $req_path) !== false)) {
          $matched_menu_id = (int)$m['id'];
          break;
        }
      }
      $rs->close();
    }
  }

  // STRICT: if no menu mapping found, deny access (fail-closed)
  if ($matched_menu_id <= 0) {
    header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden');
    echo '<link rel="stylesheet" href="/adminconsole/assets/ui.css">';
    echo '<div class="master-wrap"><div class="headbar"><h2 style="margin:0">Access denied</h2></div>';
    echo '<div class="card" style="padding:20px"><div class="badge">403 — You do not have permission to view this page.</div>';
    echo '<div style="margin-top:12px;color:#9ca3af">No menu mapping found for this page. Contact your administrator.</div></div></div>';
    exit;
  }

  // Determine current admin/user id from session or global
  $admin_id = null;
  if (session_status() === PHP_SESSION_ACTIVE) {
    if (!empty($_SESSION['admin_user_id'])) $admin_id = (int)$_SESSION['admin_user_id'];
    elseif (!empty($_SESSION['admin_id'])) $admin_id = (int)$_SESSION['admin_id'];
    elseif (!empty($_SESSION['user_id'])) $admin_id = (int)$_SESSION['user_id'];
    elseif (!empty($_SESSION['userid'])) $admin_id = (int)$_SESSION['userid'];
  }
  if (empty($admin_id) && !empty($GLOBALS['USER']) && is_array($GLOBALS['USER']) && !empty($GLOBALS['USER']['id'])) {
    $admin_id = (int)$GLOBALS['USER']['id'];
  }

  // If we cannot determine the user id, deny access
  if (empty($admin_id)) {
    header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden');
    echo '<link rel="stylesheet" href="/adminconsole/assets/ui.css">';
    echo '<div class="master-wrap"><div class="card" style="padding:20px"><h2>Access denied</h2><div class="badge" style="margin-top:10px">403 — You do not have permission to view this page.</div></div></div>';
    exit;
  }

  // First: quick check if the menu row itself has can_view = 1
  $can_view = 0;
  $stmt = @$con->prepare("SELECT can_view FROM jos_admin_menus WHERE id = ? LIMIT 1");
  if ($stmt) {
    $stmt->bind_param('i', $matched_menu_id);
    $stmt->execute();
    $stmt->bind_result($menu_can_view);
    if ($stmt->fetch()) { $can_view = (int)$menu_can_view; }
    $stmt->close();
  }

  // If menu row explicitly allows view, permit
  if ($can_view === 1) {
    return; // allowed
  }

  // Determine role_id for this admin (if available)
  $role_id = null;
  $stmt = @$con->prepare("SELECT role_id FROM jos_admin_users WHERE id = ? LIMIT 1");
  if ($stmt) {
    $stmt->bind_param('i', $admin_id);
    $stmt->execute();
    $stmt->bind_result($role_col);
    if ($stmt->fetch()) { $role_id = (int)$role_col; }
    $stmt->close();
  }

  // Role-based check
  if (!empty($role_id)) {
    $stmt = @$con->prepare("SELECT can_view FROM jos_admin_roles_menus WHERE role_id = ? AND menu_id = ? LIMIT 1");
    if ($stmt) {
      $stmt->bind_param('ii', $role_id, $matched_menu_id);
      $stmt->execute();
      $stmt->bind_result($cv);
      if ($stmt->fetch()) { $can_view = (int)$cv; }
      $stmt->close();
    }
  }

  // Per-user override check
  if ($can_view !== 1) {
    $stmt = @$con->prepare("SELECT can_view FROM jos_admin_user_menus WHERE user_id = ? AND menu_id = ? LIMIT 1");
    if ($stmt) {
      $stmt->bind_param('ii', $admin_id, $matched_menu_id);
      $stmt->execute();
      $stmt->bind_result($cv2);
      if ($stmt->fetch()) { $can_view = (int)$cv2; }
      $stmt->close();
    }
  }

  // Additional fallback table if present
  if ($can_view !== 1) {
    $stmt = @$con->prepare("SELECT can_view FROM jos_admin_menu_permissions WHERE admin_id = ? AND menu_id = ? LIMIT 1");
    if ($stmt) {
      $stmt->bind_param('ii', $admin_id, $matched_menu_id);
      $stmt->execute();
      $stmt->bind_result($cv3);
      if ($stmt->fetch()) { $can_view = (int)$cv3; }
      $stmt->close();
    }
  }

  // Final decision: if not explicitly allowed, deny
  if ($can_view !== 1) {
    header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden');
    echo '<link rel="stylesheet" href="/adminconsole/assets/ui.css">';
    echo '<div class="master-wrap">';
    echo '<div class="headbar"><h2 style="margin:0">Access denied</h2></div>';
    echo '<div class="card" style="padding:20px">';
    echo '<div class="badge">403 — You do not have permission to view this page.</div>';
    echo '<div style="margin-top:12px;color:#9ca3af">If you believe this is an error, contact your administrator.</div>';
    echo '</div></div>';
    exit;
  }

})(); // end ACL IIFE

/* ---------- Ensure DOMAIN_URL if not defined (used for images) ---------- */
if (!defined('DOMAIN_URL')) {
    define('DOMAIN_URL', '/');
}

/* ---------------- Helpers ---------------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function keep_params(array $changes=[]){
  $qs = $_GET; foreach($changes as $k=>$v){ if($v===null){unset($qs[$k]);} else {$qs[$k]=$v;} }
  $q = http_build_query($qs); return $q?('?'.$q):'';
}
function get_int($key,$default=0){ return isset($_GET[$key]) ? (int)$_GET[$key] : $default; }
function get_str($key,$default=''){ return isset($_GET[$key]) ? trim((string)$_GET[$key]) : $default; }
function fmt_date($s){ return $s ? date('d M Y', strtotime($s)) : ''; }
function fmt_dt($s){ return $s ? date('d M Y h:i A', strtotime($s)) : ''; }
function safe_date_label($raw){
  if(!$raw) return '';
  $ts = strtotime($raw);
  if($ts === false) return '';
  $y = (int)date('Y', $ts);
  if($y < 1900) return '';
  return date('d M Y', $ts);
}
function chip_list($csv){
  $csv = trim((string)$csv);
  if($csv==='') return '';
  $parts = array_filter(array_map('trim', explode(',', $csv)));
  if(!$parts) return '';
  $html = '<div style="display:flex;flex-wrap:wrap;gap:8px">';
  foreach($parts as $p){
    $html .= '<span class="badge" style="background:#0b1220;color:#cbd5e1;border:1px solid #243045">'.h($p).'</span>';
  }
  $html .= '</div>';
  return $html;
}

/* ======================================================================
   MODE: Candidate Profile Details  (?candidate={userid})
   ====================================================================== */
if (isset($_GET['candidate'])) {
  $userid = (int)$_GET['candidate'];

  $candidate_query = "
    SELECT 
        c.*,
        g.name  AS gender_name,
        COALESCE(e.name, et.name) AS experience_type_name,
        ep.name AS experience_period_name
    FROM jos_app_candidate_profile c
    LEFT JOIN jos_crm_gender g
           ON c.gender_id = g.id
    LEFT JOIN jos_crm_experience e
           ON (
                CAST(c.experience_type AS UNSIGNED) = e.id
                OR LOWER(c.experience_type) = LOWER(e.name)
              )
    LEFT JOIN jos_app_experience_list et
           ON (
                CAST(c.experience_type AS UNSIGNED) = et.id
                OR LOWER(c.experience_type) = LOWER(et.name)
              )
    LEFT JOIN jos_app_experience_list ep
           ON (
                CAST(c.experience_period AS UNSIGNED) = ep.id
                OR LOWER(c.experience_period) = LOWER(ep.name)
              )
    WHERE c.userid = ?
    LIMIT 1
  ";
  $stmt = $con->prepare($candidate_query);
  $stmt->bind_param('i', $userid);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res->fetch_assoc();
  $stmt->close();

  ob_start(); ?>
  <link rel="stylesheet" href="/adminconsole/assets/ui.css">
  <div class="master-wrap">
    <div class="headbar">
      <h2 style="margin:0"><?= h($row['candidate_name'] ?? 'Candidate') ?></h2>
      <div style="margin-left:auto;display:flex;gap:8px">
        <a class="btn secondary" href="<?= h(keep_params(['candidate'=>null])) ?>">← Back</a>
        <button class="btn secondary" onclick="window.print()">Print</button>
      </div>
    </div>

    <div class="card" style="padding:20px">
      <?php if(!$row){ ?>
        <div class="badge">No profile</div>
      <?php } else {
          $job_positions = '';
          if (!empty($row['job_position_ids'])) {
            $ids = array_filter(array_map('intval', explode(',', (string)$row['job_position_ids'])));
            if ($ids) {
              $id_list = implode(',', $ids);
              $q = "SELECT name FROM jos_crm_jobpost WHERE id IN ($id_list)";
              if ($rs = mysqli_query($con, $q)) {
                $names = [];
                while($jr = mysqli_fetch_assoc($rs)){ $names[] = $jr['name']; }
                $job_positions = implode(', ', $names);
              }
            }
          }
          $photo = isset($row['profile_photo']) ? trim((string)$row['profile_photo']) : '';
          if ($photo === '' || $photo === null) {
              $photo_url = DOMAIN_URL . 'webservices/uploads/nophoto_greyscale_circle.png';
          } elseif (stripos($photo, 'http://') === 0 || stripos($photo, 'https://') === 0) {
              $photo_url = $photo;
          } else {
              $photo_url = DOMAIN_URL . $photo;
          }
        ?>
        <div style="display:flex;align-items:center;gap:16px;margin-bottom:16px">
          <div style="height:72px;width:72px;border-radius:50%;background:#111827;overflow:hidden;display:flex;align-items:center;justify-content:center">
            <img src="<?= h($photo_url) ?>" alt="photo" style="height:100%;width:100%;object-fit:cover">
          </div>
          <div>
            <div style="font-size:18px;font-weight:700;color:#fff"><?= h($row['candidate_name']) ?></div>
            <div style="color:#9ca3af"><?= h($row['email'] ?: '') ?><?= ($row['email'] && $row['mobile_no'])?' • ':'' ?><?= h($row['mobile_no'] ?: '') ?></div>
            <?php if(!empty($job_positions)){ ?>
              <div style="margin-top:6px"><?= chip_list($job_positions) ?></div>
            <?php } ?>
          </div>
        </div>

        <div style="height:1px;background:#1f2937;margin:6px 0 16px"></div>

        <div style="display:grid;grid-template-columns:repeat(2,minmax(260px,1fr));gap:12px 24px">
          <?php
            $specs = [
              'Gender'            => $row['gender_name'] ?? '',
              'Birthdate'         => safe_date_label($row['birthdate'] ?? ''),
              'Experience Type'   => $row['experience_type_name'] ?? '',
              'Experience Period' => $row['experience_period_name'] ?? '',
              'Address'           => $row['address'] ?? '',
              'City ID'           => $row['city_id'] ?? '',
              'Locality ID'       => $row['locality_id'] ?? '',
              'Latitude'          => (isset($row['latitude']) ? trim((string)$row['latitude']) : ''),
              'Longitude'         => (isset($row['longitude']) ? trim((string)$row['longitude']) : ''),
              'Created'           => safe_date_label($row['created_at'] ?? ''),
            ];
            foreach($specs as $label=>$val){
              $val = trim((string)$val);
              if($val==='') continue;
              echo '<div style="display:flex;gap:8px"><div style="min-width:160px;color:#94a3b8">'.h($label).'</div><div style="color:#e5e7eb">'.h($val).'</div></div>';
            }
          ?>
        </div>

        <?php
          $skillsHTML = chip_list($row['skills'] ?? '');
          if($skillsHTML || !empty($row['exp_description'])){
            echo '<div style="height:1px;background:#1f2937;margin:16px 0"></div>';
          }
        ?>

        <?php if($skillsHTML){ ?>
          <div style="margin-bottom:12px">
            <div style="font-weight:600;color:#cbd5e1;margin-bottom:6px">Skills</div>
            <?= $skillsHTML ?>
          </div>
        <?php } ?>

        <?php if(!empty($row['exp_description'])){ ?>
          <div>
            <div style="font-weight:600;color:#cbd5e1;margin-bottom:6px">Profile Summary</div>
            <div style="white-space:pre-wrap;color:#e5e7eb"><?= h($row['exp_description']) ?></div>
          </div>
        <?php } ?>
      <?php } ?>
    </div>
  </div>
  <?php
  echo ob_get_clean();
  exit;
}

/* ======================================================================
   MODE: Applications list for a Standard Job  (?apps={job_id})
   Uses jos_app_applications with job_listing_type = 2
   ====================================================================== */
if (isset($_GET['apps'])) {
  $jobId = (int)$_GET['apps'];

  // Fetch job title + company label
  $stmt = $con->prepare("
      SELECT jv.id, jp.name AS job_position,
             COALESCE(jv.company_name, rp.organization_name) AS company_label
      FROM jos_app_jobvacancies jv
      LEFT JOIN jos_crm_jobpost jp ON jv.job_position_id = jp.id
      LEFT JOIN jos_app_recruiter_profile rp ON jv.recruiter_id = rp.id
      WHERE jv.id=? LIMIT 1
  ");
  $stmt->bind_param('i',$jobId);
  $stmt->execute();
  $res = $stmt->get_result();
  $job = $res->fetch_assoc();
  $stmt->close();

  // Applications + candidate basics
  $sql = "SELECT a.id,
                 a.userid,
                 a.application_date,
                 a.status_id,
                 a.interview_date_time,
                 cp.candidate_name,
                 cp.mobile_no,
                 cp.email
          FROM jos_app_applications a
          LEFT JOIN jos_app_candidate_profile cp ON cp.userid = a.userid
          WHERE a.job_listing_type = 2 AND a.job_id = ?
          ORDER BY a.application_date DESC";
  $stmt = $con->prepare($sql);
  $stmt->bind_param('i', $jobId);
  $stmt->execute();
  $apps = $stmt->get_result();
  $rows = [];
  while($r=$apps->fetch_assoc()){ $rows[]=$r; }
  $stmt->close();

  ob_start(); ?>
  <link rel="stylesheet" href="/adminconsole/assets/ui.css">
  <div class="master-wrap">
    <div class="headbar">
      <h2 style="margin:0">Applications — <?= h($job['job_position'] ?? ('Job #'.$jobId)) ?></h2>
      <div style="margin-left:auto;display:flex;gap:8px">
        <?php if(!empty($job['company_label'])){ ?>
          <span class="badge"><?= h($job['company_label']) ?></span>
        <?php } ?>
        <a class="btn secondary" href="<?= h(keep_params(['apps'=>null])) ?>">← Back</a>
      </div>
    </div>

    <div class="card">
      <div class="table-wrap">
        <table class="table">
          <thead>
            <tr>
              <th style="width:80px">ID</th>
              <th>Candidate</th>
              <th>Contact</th>
              <th>Applied On</th>
              <th>Status</th>
              <th>Interview</th>
              <th style="width:160px">Actions</th>
            </tr>
          </thead>
          <tbody>
          <?php if(!$rows){ ?>
            <tr><td colspan="7" style="text-align:center;color:#9ca3af">No applications yet.</td></tr>
          <?php } else {
            foreach($rows as $r){
              $viewCandUrl = h(keep_params(['candidate'=>(int)$r['userid']]));
              echo '<tr>';
              echo '<td>'.(int)$r['id'].'</td>';
              echo '<td>'.h($r['candidate_name'] ?: ('User #'.$r['userid'])).'</td>';
              echo '<td>';
                $contact = [];
                if(!empty($r['mobile_no'])) $contact[] = h($r['mobile_no']);
                if(!empty($r['email']))     $contact[] = h($r['email']);
                echo implode('<br>', $contact);
              echo '</td>';
              echo '<td>'.h(fmt_dt($r['application_date'])).'</td>';
              echo '<td>'.h((string)$r['status_id']).'</td>';
              echo '<td>'.h(fmt_dt($r['interview_date_time'])).'</td>';
              echo '<td><a class="btn secondary" href="'.$viewCandUrl.'" target="_blank" rel="noopener">View Candidate</a></td>';
              echo '</tr>';
            }
          } ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
  <?php
  echo ob_get_clean();
  exit;
}

/* ======================================================================
   MODE: JOB VIEW (detail) — ?view=ID  (Standard Job from jos_app_jobvacancies)
   ====================================================================== */
if (isset($_GET['view'])) {
  $id = (int)$_GET['view'];

  $sql = "SELECT 
            jv.*,
            jp.name AS job_position,
            g.name  AS gender,
            qs.name AS qualification,
            exp_from.name AS experience_from_name,
            exp_to.name   AS experience_to_name,
            sr_from.salaryrange AS salary_from_value,
            sr_to.salaryrange   AS salary_to_value,
            js.name AS job_status,
            rp.organization_name AS recruiter_org,
            rp.company_logo,
            rp.mobile_no AS recruiter_mobile_no,
            (SELECT COUNT(*) FROM jos_app_applications a WHERE a.job_listing_type=2 AND a.job_id=jv.id) AS apps_count
          FROM jos_app_jobvacancies jv
          LEFT JOIN jos_crm_jobpost jp ON jv.job_position_id = jp.id
          LEFT JOIN jos_crm_gender g ON jv.gender_id = g.id
          LEFT JOIN jos_crm_education_status qs ON jv.qualification_id = qs.id
          LEFT JOIN jos_app_experience_list exp_from ON jv.experience_from = exp_from.id
          LEFT JOIN jos_app_experience_list exp_to   ON jv.experience_to   = exp_to.id
          LEFT JOIN jos_crm_salary_range sr_from ON jv.salary_from = sr_from.id
          LEFT JOIN jos_crm_salary_range sr_to   ON jv.salary_to   = sr_to.id
          LEFT JOIN jos_app_jobstatus js ON jv.job_status_id = js.id
          LEFT JOIN jos_app_recruiter_profile rp ON jv.recruiter_id = rp.id
          WHERE jv.id = ? LIMIT 1";
  $stmt = $con->prepare($sql);
  $stmt->bind_param('i',$id);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res->fetch_assoc();
  $stmt->close();

  ob_start(); ?>
  <link rel="stylesheet" href="/adminconsole/assets/ui.css">
  <div class="master-wrap">
    <div class="headbar">
      <h2 style="margin:0"><?= h($row ? ($row['job_position'] ?: 'Job Details') : 'Job not found') ?></h2>
      <div style="margin-left:auto;display:flex;gap:8px">
        <?php if($row){ 
          $appsUrl = h(keep_params(['apps'=>(int)$row['id']]));
          ?>
          <a class="btn secondary" href="<?= $appsUrl ?>" target="_blank">View Applications (<?= (int)$row['apps_count'] ?>)</a>
        <?php } ?>
        <a class="btn secondary" href="<?=h(keep_params(['view'=>null]))?>">← Back to List</a>
        <button class="btn secondary" onclick="window.print()">Print</button>
      </div>
    </div>

    <?php if(!$row){ ?>
      <div class="card" style="padding:20px"><div class="badge">Job not found</div></div>
    <?php } else { 
        $logo = !empty($row['company_logo']) 
                  ? DOMAIN_URL.'webservices/'.$row['company_logo'] 
                  : DOMAIN_URL.'webservices/uploads/nologo.png';
        $company = $row['company_name'] ?: ($row['recruiter_org'] ?? '-');
        $posted  = safe_date_label($row['created_at'] ?? '');
    ?>
    <div class="card" style="padding:20px">

      <!-- Header -->
      <div style="display:flex;align-items:center;gap:16px;margin-bottom:16px">
        <div style="height:64px;width:64px;flex:0 0 64px;border-radius:12px;background:#111827;display:flex;align-items:center;justify-content:center;overflow:hidden">
          <?php if($logo){ ?><img src="<?=h($logo)?>" alt="logo" style="max-height:100%;max-width:100%"><?php } ?>
        </div>
        <div style="min-width:0">
          <div style="font-size:20px;font-weight:700;color:#fff;line-height:1.2"><?= h($row['job_position'] ?: '') ?></div>
          <div style="color:#9ca3af;white-space:nowrap;overflow:hidden;text-overflow:ellipsis"><?= h($company) ?></div>
          <?php if($posted){ ?><div style="color:#6b7280;font-size:12px;margin-top:2px">Posted on <?= h($posted) ?></div><?php } ?>
        </div>
        <div style="margin-left:auto;display:flex;align-items:center;gap:8px">
          <?php if(!empty($row['job_status'])){ ?>
            <span class="badge" style="background:#0b3b2a;color:#a7f3d0;border:1px solid #14532d"><?= h($row['job_status']) ?></span>
          <?php } ?>
          <span class="badge" style="background:#101a2e;border:1px solid #1f2e50;color:#cbd5e1">Applications: <?= (int)$row['apps_count'] ?></span>
        </div>
      </div>

      <div style="height:1px;background:#1f2937;margin:6px 0 16px"></div>

      <!-- Spec grid -->
      <div style="display:grid;grid-template-columns:repeat(2,minmax(260px,1fr));gap:14px 24px">
        <?php
          $specs = [
            'Gender'           => $row['gender'] ?? '',
            'Qualification'    => $row['qualification'] ?? '',
            'Experience From'  => $row['experience_from_name'] ?? '',
            'Experience To'    => $row['experience_to_name'] ?? '',
            'Salary From'      => $row['salary_from_value'] ?? '',
            'Salary To'        => $row['salary_to_value'] ?? '',
            'City ID'          => $row['city'] ?? $row['city_id'] ?? '',
            'Locality ID'      => $row['locality'] ?? $row['locality_id'] ?? '',
            'Status'           => $row['job_status'] ?? '',
          ];
          foreach($specs as $label=>$val){
            $val = trim((string)$val);
            if($val==='') continue;
            echo '<div style="display:flex;gap:8px"><div style="min-width:140px;color:#94a3b8">'.h($label).'</div><div style="color:#e5e7eb">'.h($val).'</div></div>';
          }
        ?>
      </div>

      <div style="height:1px;background:#1f2937;margin:16px 0"></div>

      <div style="display:grid;grid-template-columns:repeat(2,minmax(260px,1fr));gap:12px 24px">
        <?php
          $contact = [
            'Contact Person'   => $row['contact_person'] ?? '',
            'Contact Mobile'   => $row['contact_no'] ?? ($row['recruiter_mobile_no'] ?? ''),
            'Interview Address'=> $row['interview_address'] ?? '',
          ];
          foreach($contact as $label=>$val){
            $val = trim((string)$val);
            if($val==='') continue;
            echo '<div style="display:flex;gap:8px"><div style="min-width:140px;color:#94a3b8">'.h($label).'</div><div style="color:#e5e7eb">'.h($val).'</div></div>';
          }
        ?>
      </div>
    </div>
    <?php } ?>
  </div>
  <?php
  echo ob_get_clean();
  exit;
}

/* ======================================================================
   LIST MODE (Standard Jobs) + Applications count (listing_type = 2)
   ====================================================================== */
$page_title = 'Standard Jobs List';
$DEFAULT_PAGE_SIZE = 50;

$SQL_SELECT = 'jv.id, jv.created_at,
               jp.name AS job_position,
               COALESCE(jv.company_name, rp.organization_name) AS company_label,
               js.name AS job_status,
               (SELECT COUNT(*) FROM jos_app_applications a WHERE a.job_listing_type=2 AND a.job_id=jv.id) AS apps_count';
$SQL_FROM   = 'FROM jos_app_jobvacancies jv
               LEFT JOIN jos_crm_jobpost jp ON jv.job_position_id = jp.id
               LEFT JOIN jos_app_jobstatus js ON jv.job_status_id = js.id
               LEFT JOIN jos_app_recruiter_profile rp ON jv.recruiter_id = rp.id';
$SORT_MAP = [
  'newest'=>'ORDER BY jv.id DESC',
  'oldest'=>'ORDER BY jv.id ASC',
  'name_asc'=>'ORDER BY jp.name ASC',
  'name_desc'=>'ORDER BY jp.name DESC'
];

/* -------- Filters (now with dynamic Job Status) -------- */
$q            = get_str('q',''); // searches job position & company/org
$status_id    = get_int('status_id', 0); // from dropdown
$created_from = get_str('created_from','');
$created_to   = get_str('created_to','');
$sort         = get_str('sort','newest');
$view         = get_str('view','last50');
$page         = max(1,get_int('page',1));
$per_page     = ($view==='all')?1000:$DEFAULT_PAGE_SIZE;
$offset       = ($page-1)*$per_page;

$where=[];$types='';$params=[];
if($q!==''){
  $where[]="(COALESCE(jv.company_name, rp.organization_name) LIKE CONCAT('%',?,'%') OR jp.name LIKE CONCAT('%',?,'%'))";
  $types.='ss'; $params[]=$q; $params[]=$q;
}
if ($status_id > 0) {
  $where[] = "jv.job_status_id = ?";
  $types .= 'i'; $params[] = $status_id;
}
if ($created_from!=='' && preg_match('/^\d{4}-\d{2}-\d{2}$/',$created_from)){
  $where[]="DATE(jv.created_at)>=?"; $types.='s'; $params[]=$created_from;
}
if ($created_to!=='' && preg_match('/^\d{4}-\d{2}-\d{2}$/',$created_to)){
  $where[]="DATE(jv.created_at)<=?"; $types.='s'; $params[]=$created_to;
}
$sql_where=$where?(' WHERE '.implode(' AND ',$where)):'';
$order=$SORT_MAP[$sort]??reset($SORT_MAP);

/* count */
$sql_count="SELECT COUNT(*) ".$SQL_FROM.$sql_where;
$stmt=$con->prepare($sql_count);
if($types){$stmt->bind_param($types,...$params);}
$stmt->execute();$stmt->bind_result($total);$stmt->fetch();$stmt->close();

/* pagination */
$pages=($view!=='all')?max(1,ceil($total/$per_page)):1;
if($page>$pages){$page=$pages;$offset=($page-1)*$per_page;}

/* main */
$sql="SELECT ".$SQL_SELECT." ".$SQL_FROM.$sql_where." ".$order." ".($view==='all'?"":" LIMIT $per_page OFFSET $offset");
$stmt=$con->prepare($sql);
if($types){$stmt->bind_param($types,...$params);}
$stmt->execute();$res=$stmt->get_result();

/* Fetch statuses for dropdown (display_status=1) */
$statuses=[];
if ($rs = $con->query("SELECT id, name FROM jos_app_jobstatus WHERE display_status=1 ORDER BY orderby, id")) {
  while($r = $rs->fetch_assoc()){ $statuses[] = $r; }
}

/* columns */
$COLUMNS = [
  ['label'=>'SR No','width'=>'70px','render'=>function($row,$sr){echo (int)$sr;}],
  ['label'=>'Job / Company','render'=>function($row){
    echo '<div>'.h($row['job_position'] ?? '').'</div>';
    if(!empty($row['company_label'])){
      echo '<div style="font-size:12px;color:#9ca3af">'.h($row['company_label']).'</div>';
    }
  }],
  ['label'=>'Status','render'=>function($row){echo h($row['job_status']??'');}],
  ['label'=>'Posted On','render'=>function($row){echo h(fmt_date($row['created_at']));}],
  ['label'=>'Applications','render'=>function($row){ echo (int)($row['apps_count'] ?? 0); }],
  ['label'=>'Actions','render'=>function($row){
    $viewUrl  = h(keep_params(['view'=>(int)$row['id']]));
    $appsUrl  = h(keep_params(['apps'=>(int)$row['id']]));
    echo '<div style="display:flex;gap:6px;flex-wrap:wrap">';
    echo   '<a class="btn secondary" href="'.$viewUrl.'" target="_blank">View Job</a>';
    echo   '<a class="btn secondary" href="'.$appsUrl.'" target="_blank">View Applications ('.(int)$row['apps_count'].')</a>';
    echo '</div>';
  }]
];

/* render */
ob_start(); ?>
<link rel="stylesheet" href="/adminconsole/assets/ui.css">
<div class="master-wrap">
  <div class="headbar"><h2 style="margin:0"><?=h($page_title)?></h2></div>

  <div class="card">
    <form method="get" class="toolbar" style="gap:10px;flex-wrap:wrap">
      <input class="inp" type="text" name="q" value="<?=h($q)?>" placeholder="Search job/company..." style="min-width:240px">

      <select class="inp" name="status_id">
        <option value="0" <?= $status_id===0?'selected':''; ?>>All Status</option>
        <?php foreach($statuses as $st): ?>
          <option value="<?=h($st['id'])?>" <?= $status_id===$st['id']?'selected':''; ?>><?=h($st['name'])?></option>
        <?php endforeach; ?>
      </select>

      <input class="inp" type="date" name="created_from" value="<?=h($created_from)?>">
      <input class="inp" type="date" name="created_to" value="<?=h($created_to)?>">

      <select class="inp" name="sort">
        <?php foreach($SORT_MAP as $k=>$v): ?>
        <option value="<?=$k?>" <?=$sort===$k?'selected':''?>><?=h($k)?></option>
        <?php endforeach;?>
      </select>

      <button class="btn primary" type="submit">Apply</button>

      <div style="flex:1"></div>
      <a class="btn secondary" href="<?=h(keep_params(['view'=>'last50','page'=>1]))?>">Last <?=$DEFAULT_PAGE_SIZE?></a>
      <a class="btn secondary" href="<?=h(keep_params(['view'=>'all','page'=>1]))?>">View All</a>
    </form>

    <div style="display:flex;align-items:center;gap:12px;margin:8px 0 12px">
      <span class="badge">Total: <?= (int)$total ?></span>
      <span class="badge">Showing: <?= ($view==='all') ? 'All' : ($res->num_rows) ?></span>
      <?php if($view!=='all'){ ?>
        <div style="margin-left:auto;display:flex;gap:6px;align-items:center">
          <?php if($page>1){ ?><a class="btn secondary" href="<?=h(keep_params(['page'=>$page-1]))?>">‹ Prev</a><?php } ?>
          <span>Page <?= (int)$page ?> / <?= (int)$pages ?></span>
          <?php if($page<$pages){ ?><a class="btn secondary" href="<?=h(keep_params(['page'=>$page+1]))?>">Next ›</a><?php } ?>
        </div>
      <?php } ?>
    </div>

    <div class="table-wrap">
      <table class="table">
        <thead>
          <tr>
            <?php foreach($COLUMNS as $col): ?>
              <th<?=isset($col['width'])?' style="width:'.$col['width'].';"':''?>><?=h($col['label'])?></th>
            <?php endforeach; ?>
          </tr>
        </thead>
        <tbody>
        <?php
          $srStart = ($view==='all') ? 1 : ($offset+1);
          $sr = $srStart;
          while($row = $res->fetch_assoc()):
            echo '<tr>';
            foreach($COLUMNS as $col){
              echo '<td>';
              $col['render']($row, $sr);
              echo '</td>';
            }
            echo '</tr>';
            $sr++;
          endwhile;
          $stmt->close();

          if ($sr === $srStart){
            echo '<tr><td colspan="'.count($COLUMNS).'" style="text-align:center;color:#9ca3af">No records found.</td></tr>';
          }
        ?>
        </tbody>
      </table>
    </div>

    <?php if($view!=='all'){ ?>
      <div style="display:flex;gap:8px;justify-content:flex-end;margin-top:12px">
        <?php if($page>1){ ?><a class="btn secondary" href="<?=h(keep_params(['page'=>$page-1]))?>">‹ Prev</a><?php } ?>
        <span class="badge">Page <?= (int)$page ?> / <?= (int)$pages ?></span>
        <?php if($page<$pages){ ?><a class="btn secondary" href="<?=h(keep_params(['page'=>$page+1]))?>">Next ›</a><?php } ?>
      </div>
    <?php } ?>
  </div>
</div>
<?php
echo ob_get_clean();
