<?php
require_once __DIR__ . '/../includes/auth.php';
require_login();
require_once __DIR__ . '/../includes/initialize.php';

/* ---------- page config (fallback) ---------- */
$page_title = 'Subscription Plans';
$TABLE      = 'jos_app_subscription_plans';
$LOGTABLE   = 'jos_app_usersubscriptionlog';
$USERSTAB   = 'jos_app_users';
$FEATTABLE  = 'jos_app_userfeatures';
$PTTABLE    = 'jos_app_profile_types';
$MENUTABLE  = 'jos_admin_menus'; // Menu master table

/* ---------- helpers ---------- */
function keep_params(array $changes = []) {
  $qs = $_GET; foreach ($changes as $k=>$v) { if ($v===null) unset($qs[$k]); else $qs[$k]=$v; }
  return '?' . http_build_query($qs);
}
function clean($v){ return trim((string)$v); }
function col_exists($con,$table,$col){
  $r=@mysqli_query($con,"SHOW COLUMNS FROM `".mysqli_real_escape_string($con,$table)."` LIKE '".mysqli_real_escape_string($con,$col)."'");
  return ($r && mysqli_num_rows($r)>0);
}
function any_col($con,$table,$candidates){
  foreach ($candidates as $c){ if (col_exists($con,$table,$c)) return $c; }
  return null;
}
function ensure_schema($con,$table){
  if(!col_exists($con,$table,'status')){
    mysqli_query($con,"ALTER TABLE `".mysqli_real_escape_string($con,$table)."` ADD `status` TINYINT(1) NOT NULL DEFAULT 1");
  }
  if(!col_exists($con,$table,'orderby')){
    mysqli_query($con,"ALTER TABLE `".mysqli_real_escape_string($con,$table)."` ADD `orderby` INT NOT NULL DEFAULT 0");
  }
}
function back_to_list($msg){ header('Location: '.keep_params(['add'=>null,'edit'=>null,'ok'=>$msg])); exit; }

/* ---------- menu/title and permissions loader (uses menu_link & menu_name) ---------- */
function fetch_menu_info($con, $script_name) {
  global $MENUTABLE, $page_title;
  $out = ['title'=>$page_title, 'perms'=>['view'=>true,'add'=>true,'edit'=>true,'delete'=>true]];

  // Verify menu table exists
  $ok = @mysqli_query($con, "SHOW TABLES LIKE '".mysqli_real_escape_string($con,$MENUTABLE)."'");
  if (!$ok || mysqli_num_rows($ok)===0) return $out;

  // require menu_link column to be present to perform lookup
  if (!col_exists($con, $MENUTABLE, 'menu_link')) return $out;

  $script_base = basename($script_name);
  $sql = "SELECT * FROM `".mysqli_real_escape_string($con,$MENUTABLE)."` WHERE `menu_link` LIKE CONCAT('%',?,'%') AND `status`=1 LIMIT 1";
  $stmt = @$con->prepare($sql);
  if (!$stmt) return $out;

  $stmt->bind_param("s", $script_base);
  $stmt->execute();
  $res = $stmt->get_result();
  if ($res && $row = $res->fetch_assoc()) {
    // use menu_name as page title if present
    if (isset($row['menu_name']) && trim($row['menu_name'])!=='') $out['title'] = $row['menu_name'];

    // default permissions: true (no permission columns in your table currently)
    // If you add permission columns later, extend this block to read them.
    foreach (['view','add','edit','delete'] as $k) $out['perms'][$k] = true;
  }

  $stmt->close();
  return $out;
}

/* ---------- convenience can() using menu perms ---------- */
$menu_info = fetch_menu_info($con, $_SERVER['PHP_SELF'] ?? __FILE__);
$page_title = $menu_info['title'] ?? $page_title;
function can_action($action) {
  global $menu_info;
  $action = strtolower($action);
  return !empty($menu_info['perms'][$action]);
}

/* ---------- profile types & features helpers (unchanged) ---------- */
function fetch_profile_types($con){
  $label = any_col($con,'jos_app_profile_types',['name','title','label','profile_type','type_name','typename']);
  if (!$label) return [1=>'Job Seeker',2=>'Recruiter',3=>'Promoter'];
  $order = col_exists($con,'jos_app_profile_types','orderby') ? "`orderby`, $label" : $label;
  $sql = "SELECT id,$label AS label FROM jos_app_profile_types".(col_exists($con,'jos_app_profile_types','status')?' WHERE status=1':'')." ORDER BY $order";
  $rs = mysqli_query($con,$sql); $out=[];
  if($rs){ while($r=mysqli_fetch_assoc($rs)){ $out[(int)$r['id']]=$r['label']; } }
  return $out ?: [1=>'Job Seeker',2=>'Recruiter',3=>'Promoter'];
}
function fetch_features_flat($con){
  $label = any_col($con,'jos_app_userfeatures',['name','title','label']); if(!$label) $label='id';
  $ptcol = col_exists($con,'jos_app_userfeatures','profile_type') ? 'profile_type' : '0 AS profile_type';
  $order = [];
  if (col_exists($con,'jos_app_userfeatures','profile_type')) $order[]='profile_type';
  if (col_exists($con,'jos_app_userfeatures','orderby'))      $order[]='`orderby`';
  $order[] = $label;
  $sql = "SELECT id,$label AS name,$ptcol FROM jos_app_userfeatures ".(col_exists($con,'jos_app_userfeatures','status')?'WHERE status=1 ':'')."ORDER BY ".implode(', ',$order);
  $rs = mysqli_query($con,$sql); $out=[];
  if($rs){ while($r=mysqli_fetch_assoc($rs)){ $out[]=['id'=>(int)$r['id'],'name'=>$r['name'],'pt'=>(int)$r['profile_type']]; } }
  return $out;
}

/* ---------- bootstrap: ensure schema ---------- */
ensure_schema($con,$TABLE);
$err = clean($_GET['err'] ?? '');
$ok  = clean($_GET['ok'] ?? '');

/* ---------- POST handling with permission enforcement ---------- */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!verify_csrf($_POST['csrf'] ?? null)) {
    $err = 'Invalid request. Please refresh and try again.';
  } else {
    if (isset($_POST['delete'])) {
      if (!can_action('delete')) { $err = 'Permission denied.'; }
      else {
        $id = (int)($_POST['id'] ?? 0);
        if ($id>0) {
          $stmt=$con->prepare("DELETE FROM `$TABLE` WHERE id=?");
          $stmt->bind_param("i",$id); $stmt->execute(); $stmt->close();
          back_to_list('Deleted successfully');
        } else { $err='Invalid row selected.'; }
      }
    }
    if (isset($_POST['save'])) {
      if (!can_action('edit') && !can_action('add')) { $err = 'Permission denied.'; }
      else {
        $id              = (int)($_POST['id'] ?? 0);
        $plan_name       = clean($_POST['plan_name'] ?? '');
        $profile_type    = (int)($_POST['profile_type'] ?? 0);
        $validity_months = (int)($_POST['validity_months'] ?? 0);
        $walkin_limit    = (int)($_POST['walkin_limit'] ?? 0);
        $vacancy_limit   = (int)($_POST['vacancy_limit'] ?? 0);
        $gst             = (float)($_POST['gst'] ?? 0);
        $final_amount    = (float)($_POST['final_amount'] ?? 0);
        $feat_ids_arr = array_map('intval', $_POST['features_ids'] ?? []);
        $feat_ids_arr = array_values(array_unique(array_filter($feat_ids_arr, fn($x)=>$x>0)));
        sort($feat_ids_arr);
        $features_csv   = implode(',', $feat_ids_arr);
        $plan_status    = (int)($_POST['plan_status'] ?? 0);
        $premium_status = (int)($_POST['premium_status'] ?? 0);
        $display_status = (int)($_POST['display_status'] ?? 0);
        $plan_access    = clean($_POST['plan_access'] ?? '');
        $status         = (int)($_POST['status'] ?? 1);
        $orderby        = (int)($_POST['orderby'] ?? 0);

        if ($plan_name==='') { $err='Plan name is required.'; }
        if (!$err && $profile_type<=0){ $err='Please choose a Profile Type.'; }

        if (!$err){
          $stmt=$con->prepare("SELECT id FROM `$TABLE` WHERE LOWER(plan_name)=LOWER(?) AND profile_type=? AND id<>? LIMIT 1");
          $stmt->bind_param("sii",$plan_name,$profile_type,$id);
          $stmt->execute(); $stmt->store_result();
          if ($stmt->num_rows>0) $err='Duplicate plan for the same profile type.';
          $stmt->close();
        }

        if (!$err){
          if ($orderby<=0){
            $r = mysqli_query($con,"SELECT COALESCE(MAX(orderby),0)+1 AS nxt FROM `$TABLE`");
            $orderby = (int)mysqli_fetch_assoc($r)['nxt'];
          }
          if ($id>0){
            if (!can_action('edit')) { $err = 'Permission denied for edit.'; }
            else {
              $sql = "UPDATE `$TABLE` SET
                        plan_name=?, profile_type=?, validity_months=?, walkin_limit=?, vacancy_limit=?, gst=?, final_amount=?, features=?, plan_status=?, premium_status=?, display_status=?, plan_access=?, status=?, orderby=?
                      WHERE id=?";
              $stmt=$con->prepare($sql);
              $stmt->bind_param("siiiiiddsiiisii",
                $plan_name,$profile_type,$validity_months,$walkin_limit,$vacancy_limit,
                $gst,$final_amount,$features_csv,$plan_status,$premium_status,$display_status,
                $plan_access,$status,$orderby,$id
              );
            }
          } else {
            if (!can_action('add')) { $err = 'Permission denied for add.'; }
            else {
              $sql = "INSERT INTO `$TABLE`
                        (plan_name, profile_type, validity_months, walkin_limit, vacancy_limit, gst, final_amount, features, plan_status, premium_status, display_status, plan_access, status, orderby, created_at)
                      VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,NOW())";
              $stmt=$con->prepare($sql);
              $stmt->bind_param("siiiiiddsiiisii",
                $plan_name,$profile_type,$validity_months,$walkin_limit,$vacancy_limit,
                $gst,$final_amount,$features_csv,$plan_status,$premium_status,$display_status,
                $plan_access,$status,$orderby
              );
            }
          }
          if(!$stmt){ $err='Failed to prepare statement.'; }
          else {
            if(!$stmt->execute()) $err='Save failed: '.$stmt->error;
            $stmt->close();
          }
          if(!$err) back_to_list('Saved successfully');
        }
      }
    }
  }
}

/* ---------- mode ---------- */
$mode = (isset($_GET['add']) || isset($_GET['edit'])) ? 'form' : 'list';

/* ---------- UI ---------- */
ob_start();
?>
<link rel="stylesheet" href="/adminconsole/assets/ui.css">

<!-- small local CSS to match card + top-right back button + Save in header -->
<style>
.card { background: rgba(10,15,25,0.8); border-radius:12px; padding:20px; box-sizing:border-box; }
.card-head { display:flex; align-items:center; justify-content:space-between; margin-bottom:14px; gap:12px; }
.card-head .title { font-size:1.2rem; font-weight:600; margin:0; color: #e6eef6; }
.header-actions { display:flex; gap:8px; align-items:center; }
.btn-back { background:#2b3947; color:#fff; padding:8px 14px; border-radius:10px; border:1px solid rgba(255,255,255,0.08); text-decoration:none; font-size:0.95rem; transition:all .12s ease; }
.btn-back:hover { transform:translateY(-1px); }
.btn-save-header { background:#009c5f; color:#fff; padding:8px 14px; border-radius:8px; border:none; cursor:pointer; }
.form-grid-3 { display:grid; grid-template-columns:repeat(3,minmax(0,1fr)); gap:12px; }
@media(max-width:950px){ .form-grid-3{grid-template-columns:repeat(2,minmax(0,1fr));} }
@media(max-width:600px){ .form-grid-3{grid-template-columns:1fr;} .toolbar { flex-direction:column; align-items:stretch; } }
.toolbar { display:flex; gap:12px; align-items:center; flex-wrap:wrap; }
.toolbar .right { margin-left:auto; display:flex; gap:8px; align-items:center; }
.tag-cloud { display:flex; flex-wrap:wrap; gap:8px; background:#0b1220; border:1px solid #334155; border-radius:12px; padding:10px; max-height:260px; overflow:auto; }
.tag-cloud .btn { padding:7px 10px; border-radius:10px; }
.card .inp, .card input[type="text"], .card input[type="number"], .card select { box-sizing:border-box; width:100%; }
.flash-err { background:#4d2222; color:#ffdede; padding:8px 10px; border-radius:8px; }
.flash-ok  { background:#1f3b27; color:#c8f6d1; padding:8px 10px; border-radius:8px; }
</style>

<div class="master-wrap">
  <div class="headbar" style="display:flex;align-items:center;justify-content:space-between">
    <h2 style="margin:0"><?=htmlspecialchars($page_title)?></h2>
    <div style="display:flex;gap:10px;align-items:center">
      <?php if($ok): ?><div class="flash-ok"><?=htmlspecialchars($ok)?></div><?php endif; ?>
      <?php if($err): ?><div class="flash-err"><?=htmlspecialchars($err)?></div><?php endif; ?>
    </div>
  </div>

<?php if($mode==='form'): ?>
  <?php
  $id = (int)($_GET['edit'] ?? 0);
  $row = [
    'plan_name'=>'','profile_type'=>0,'validity_months'=>0,'walkin_limit'=>0,'vacancy_limit'=>0,'gst'=>0,
    'final_amount'=>0,'features'=>'','plan_status'=>0,'premium_status'=>0,'display_status'=>0,'plan_access'=>'',
    'status'=>1,'orderby'=>0
  ];
  if ($id>0){
    $q=$con->prepare("SELECT * FROM `$TABLE` WHERE id=? LIMIT 1");
    $q->bind_param("i",$id); $q->execute();
    $res=$q->get_result(); if($res->num_rows) $row=$res->fetch_assoc();
    $q->close();
  }
  $pt_list  = fetch_profile_types($con);
  $features = fetch_features_flat($con);

  $selected = array_map(
      'intval',
      array_filter(
          explode(',', (string)($row['features'] ?? '')),
          fn($x) => $x !== ''
      )
  );
  $sel_map  = array_fill_keys($selected,true);
  ?>
  <div class="card">
    <div class="card-head">
      <h3 class="title"><?= $id>0 ? 'Edit Plan' : 'Add Plan' ?></h3>

      <div class="header-actions">
        <?php if ((($id>0 && can_action('edit')) || ($id===0 && can_action('add')))): ?>
          <!-- header save triggers hidden submit inside the form -->
          <button type="button" class="btn-save-header" id="headerSaveBtn">Save</button>
        <?php endif; ?>
        <a class="btn-back" href="<?= keep_params(['add'=>null,'edit'=>null]) ?>">Back to List</a>
      </div>
    </div>

    <form method="post" id="planForm">
      <input type="hidden" name="csrf" value="<?=htmlspecialchars(csrf_token())?>">
      <input type="hidden" name="id" value="<?=$id?>">

      <!-- hidden actual submit button clicked by header save -->
      <button type="submit" name="save" value="1" id="hiddenSaveBtn" style="display:none"></button>

      <div class="form-grid-3">
        <label>Plan Name*<br><input class="inp" name="plan_name" value="<?=htmlspecialchars($row['plan_name'])?>" required></label>
        <label>Profile Type<br>
          <select class="inp" name="profile_type" id="profile_type" required>
            <option value="">-- Select --</option>
            <?php foreach($pt_list as $pid=>$pname): ?>
              <option value="<?=$pid?>" <?=$pid==(int)$row['profile_type']?'selected':''?>><?=htmlspecialchars($pname)?></option>
            <?php endforeach; ?>
          </select>
        </label>
        <label>Validity (months)<br><input class="inp" type="number" name="validity_months" value="<?= (int)$row['validity_months'] ?>"></label>

        <label>Premium Post Job Limit<br><input class="inp" type="number" name="walkin_limit" value="<?= (int)$row['walkin_limit'] ?>"></label>
        <label>Standard Job Post Limit<br><input class="inp" type="number" name="vacancy_limit" value="<?= (int)$row['vacancy_limit'] ?>"></label>
        <label>Final Amount<br><input class="inp" type="number" step="0.01" name="final_amount" value="<?= htmlspecialchars($row['final_amount']) ?>"></label>

        <label>GST %<br><input class="inp" type="number" step="0.01" name="gst" value="<?= htmlspecialchars($row['gst']) ?>"></label>
        <label>Order Level<br><input class="inp" type="number" name="orderby" value="<?= (int)$row['orderby'] ?>"></label>
        <label>Display Status (int)<br><input class="inp" type="number" name="display_status" value="<?= (int)$row['display_status'] ?>"></label>

        <label>Plan Status (0/1)<br><input class="inp" type="number" name="plan_status" value="<?= (int)$row['plan_status'] ?>"></label>
        <label>Premium Status (0/1)<br><input class="inp" type="number" name="premium_status" value="<?= (int)$row['premium_status'] ?>"></label>
        <label>Plan Access (text)<br><input class="inp" name="plan_access" value="<?= htmlspecialchars($row['plan_access']) ?>"></label>
      </div>

      <div class="toolbar" style="margin:16px 0 8px">
        <strong>Features</strong>
        <span class="hint" id="featCount" style="margin-left:8px"></span>
      </div>

      <div id="featureCloud" class="tag-cloud">
        <?php foreach($features as $f):
          $isSel = isset($sel_map[$f['id']]); ?>
          <button type="button"
                  class="btn <?=$isSel?'success':'secondary'?> feat-chip"
                  data-id="<?=$f['id']?>" data-pt="<?=$f['pt']?>"
                  aria-pressed="<?=$isSel?'true':'false'?>">
            <?=htmlspecialchars($f['name'])?><?= $f['pt'] ? " (PT ".$f['pt'].")" : "" ?>
          </button>
          <?php if($isSel): ?>
            <input type="hidden" name="features_ids[]" value="<?=$f['id']?>">
          <?php endif; ?>
        <?php endforeach; ?>
      </div>

      <div class="toolbar" style="margin-top:14px">
        <label>Status<br>
          <select class="inp" name="status">
            <option value="1" <?=$row['status']?'selected':''?>>Active</option>
            <option value="0" <?=!$row['status']?'selected':''?>>Inactive</option>
          </select>
        </label>

        <!-- bottom toolbar no visible Save (header Save handles submission) -->
        <div class="right"></div>
      </div>
    </form>
  </div>

  <script>
    (function(){
      // header Save button submits the hidden submit inside the form
      const headerSave = document.getElementById('headerSaveBtn');
      if (headerSave) {
        headerSave.addEventListener('click', function(){
          const hidden = document.getElementById('hiddenSaveBtn');
          if (hidden) hidden.click();
        });
      }

      const ptSel = document.getElementById('profile_type');
      const cloud = document.getElementById('featureCloud');
      const form  = document.getElementById('planForm');
      const countEl = document.getElementById('featCount');

      function syncCount(){
        const chips = cloud.querySelectorAll('button.feat-chip[aria-pressed="true"]');
        countEl.textContent = '('+chips.length+' selected)';
      }

      function ensureHidden(id, on){
        const name = 'features_ids[]';
        const existing = [...form.querySelectorAll('input[type="hidden"][name="'+name+'"]')];
        const found = existing.find(i => i.value === String(id));
        if (on && !found){
          const h = document.createElement("input");
          h.type='hidden'; h.name=name; h.value=String(id);
          form.appendChild(h);
        }
        if (!on && found){ found.remove(); }
      }

      // tag toggle
      cloud.addEventListener('click', (e)=>{
        const btn = e.target.closest('.feat-chip'); if(!btn) return;
        const pressed = btn.getAttribute('aria-pressed') === 'true';
        const now = !pressed;
        btn.setAttribute('aria-pressed', now ? 'true' : 'false');
        btn.classList.toggle('success', now);
        btn.classList.toggle('secondary', !now);
        ensureHidden(btn.dataset.id, now);
        syncCount();
      });

      function filterByPT(){
        const pt = parseInt(ptSel.value || '0', 10);
        cloud.querySelectorAll('.feat-chip').forEach(btn=>{
          const fpt = parseInt(btn.dataset.pt || '0', 10);
          btn.style.display = (!pt || fpt===pt) ? '' : 'none';
          if (pt && fpt!==pt && btn.getAttribute('aria-pressed')==='true'){
            btn.setAttribute('aria-pressed','false'); btn.classList.remove('success'); btn.classList.add('secondary');
            ensureHidden(btn.dataset.id,false);
          }
        });
        syncCount();
      }
      if (ptSel) ptSel.addEventListener('change', filterByPT);

      // initial
      filterByPT();
      syncCount();
    })();
  </script>

<?php else: // list view (keeps original features) ?>

  <?php
  // filters
  $q      = clean($_GET['q'] ?? '');
  $level  = (isset($_GET['level']) && $_GET['level']!=='') ? (int)$_GET['level'] : '';
  $sort   = clean($_GET['sort'] ?? 'order_asc');
  $view   = clean($_GET['view'] ?? 'last50');
  $no_rec = (int)($_GET['no_records'] ?? 0);

  $where = " WHERE 1=1 "; $binds=[]; $types='';
  if($q!==''){ $where .= " AND (plan_name LIKE CONCAT('%',?,'%')) "; $types.='s'; $binds[]=$q; }
  if($level!==''){ $where .= " AND orderby=? "; $types.='i'; $binds[]=(int)$level; }

  $order_sql = " ORDER BY orderby ASC, id DESC ";
  if($sort==='order_desc') $order_sql=" ORDER BY orderby DESC, id DESC ";
  if($sort==='name_asc')   $order_sql=" ORDER BY plan_name ASC ";
  if($sort==='name_desc')  $order_sql=" ORDER BY plan_name DESC ";
  if($sort==='id_desc')    $order_sql=" ORDER BY id DESC ";
  if($sort==='id_asc')     $order_sql=" ORDER BY id ASC ";

  $limit_sql = ($view==='all') ? "" : " LIMIT 50 ";
  $sql = "SELECT id, plan_name, profile_type, validity_months, final_amount, orderby, status, features
          FROM `$TABLE` $where $order_sql $limit_sql";
  $stmt = $con->prepare($sql);
  if($stmt){
    if($types) $stmt->bind_param($types, ...$binds);
    $stmt->execute();
    $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
  } else { $rows = []; }

  $pt_names = fetch_profile_types($con);

  // usage counts
  $ids = array_column($rows,'id'); $counts = [];
  if ($ids) {
    $in = implode(',', array_map('intval',$ids));
    $q1 = mysqli_query($con,"SELECT plan_id AS pid, COUNT(*) c FROM `$LOGTABLE` WHERE plan_id IN ($in) GROUP BY plan_id");
    while($r=mysqli_fetch_assoc($q1)){ $counts[(int)$r['pid']] = ((int)($counts[(int)$r['pid']] ?? 0)) + (int)$r['c']; }
    if (col_exists($con,$USERSTAB,'active_plan_id')) {
      $q2 = mysqli_query($con,"SELECT active_plan_id AS pid, COUNT(*) c FROM `$USERSTAB` WHERE active_plan_id IN ($in) GROUP BY active_plan_id");
      while($r=mysqli_fetch_assoc($q2)){ $counts[(int)$r['pid']] = ((int)($counts[(int)$r['pid']] ?? 0)) + (int)$r['c']; }
    }
  }
  if ($no_rec) {
    $rows = array_values(array_filter($rows, function($r) use($counts){ return (int)($counts[(int)$r['id']] ?? 0)===0; }));
  }
  ?>
  <div class="card">
    <div class="card-head">
      <h3 class="title">Subscription Plans</h3>
      <div>
        <?php if (can_action('add')): ?>
          <a class="btn-back" href="<?=keep_params(['add'=>1])?>">Add New</a>
        <?php endif; ?>
      </div>
    </div>

    <div class="table-wrap" style="max-height:70vh">
      <table class="table">
        <thead><tr>
          <th style="width:70px">SR No</th>
          <th>Name</th>
          <th style="width:180px">Profile Type</th>
          <th style="width:120px">Validity (m)</th>
          <th style="width:120px">Final Amount</th>
          <th style="width:120px">Order Level</th>
          <th style="width:110px">Features</th>
          <th style="width:110px">Usage</th>
          <th style="width:110px">Status</th>
          <th style="width:160px">Actions</th>
        </tr></thead>
        <tbody>
          <?php if(!$rows): ?>
            <tr><td colspan="10" class="muted">No plans found.</td></tr>
          <?php endif; ?>
          <?php $sr=1; foreach($rows as $r): $featCount = ($r['features']!=='') ? count(array_filter(explode(',',$r['features']))) : 0; ?>
            <tr>
              <td><?= $sr++ ?></td>
              <td><?= htmlspecialchars($r['plan_name']) ?></td>
              <td><?= htmlspecialchars($pt_names[(int)$r['profile_type']] ?? (string)(int)$r['profile_type']) ?></td>
              <td><?= (int)$r['validity_months'] ?></td>
              <td><?= number_format((float)$r['final_amount'],2) ?></td>
              <td><?= (int)$r['orderby'] ?></td>
              <td><?= (int)$featCount ?></td>
              <td><?= (int)($counts[(int)$r['id']] ?? 0) ?></td>
              <td><?= $r['status'] ? '<span class="badge green">Active</span>' : '<span class="badge red">Inactive</span>' ?></td>
              <td>
                <?php if (can_action('edit')): ?>
                  <a class="btn secondary" href="<?=keep_params(['edit'=>$r['id']])?>">Edit</a>
                <?php endif; ?>
                <?php if (can_action('delete')): ?>
                  <form method="post" style="display:inline" onsubmit="return confirm('Delete this plan?')">
                    <input type="hidden" name="csrf" value="<?=htmlspecialchars(csrf_token())?>">
                    <input type="hidden" name="id" value="<?=$r['id']?>">
                    <button class="btn danger" name="delete" value="1">Delete</button>
                  </form>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>

<?php endif; ?>

</div> 
<?php echo ob_get_clean(); ?>
