<?php
@ini_set('display_errors','1'); @error_reporting(E_ALL);

require_once __DIR__ . '/../includes/auth.php';  require_login();
require_once __DIR__ . '/../includes/initialize.php'; // $con (mysqli)

/* ===========================================================
  Coupon Codes Master (Enhanced)
  Table: jos_app_couponcodes
  Fields used here:
    - coupon_code (unique label)
    - discount_type (percentage|flat)
    - discount_value (decimal)
    - expiry_date (YYYY-MM-DD)
    - profile_type_id (FK -> jos_app_profile_types.id)
    - max_usage_limit (int)
    - used_count (int, read-only on UI)
    - status (1/0)
    - created_by_userid, created_at, updated_at (auto)
   =========================================================== */

$DEFAULT_PAGE_TITLE = 'Coupon Codes';
$TABLE      = 'jos_app_couponcodes';
$LABEL_COL  = 'coupon_code';
$STATUS_COL = 'status';
$ORDER_COL  = 'id'; // sorting select keeps Skills Master options

/* ---------- helpers ---------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function keep_params(array $changes = []){
  $qs = $_GET; foreach ($changes as $k=>$v){ if ($v===null) unset($qs[$k]); else $qs[$k] = $v; }
  $q = http_build_query($qs); return $q ? ('?'.$q) : '';
}
function clean($v){ return trim((string)$v); }

/* SHOW COLUMNS cannot be prepared; escape and run directly. */
function col_exists(mysqli $con, string $table, string $col): bool {
  $table = preg_replace('/[^a-zA-Z0-9_]/','', $table);
  $col   = $con->real_escape_string($col);
  $sql   = "SHOW COLUMNS FROM `{$table}` LIKE '{$col}'";
  $res   = $con->query($sql);
  return ($res && $res->fetch_assoc()) ? true : false;
}

/* ensure schema (status/order columns) — safe if they already exist */
function ensure_schema(mysqli $con, string $table, string $statusCol, string $orderCol){
  if (!col_exists($con,$table,$statusCol))  { $con->query("ALTER TABLE `{$table}` ADD `{$statusCol}`  TINYINT DEFAULT 1"); }
  if (!col_exists($con,$table,$orderCol) && strtolower($orderCol)!=='id'){
    $con->query("ALTER TABLE `{$table}` ADD `{$orderCol}` INT DEFAULT 0");
    $con->query("ALTER TABLE `{$table}` MODIFY `{$orderCol}` INT DEFAULT 0");
  }
}

/* CSRF helpers (define if missing) */
if (!function_exists('csrf_token')) {
  function csrf_token(){
    if (session_status()!==PHP_SESSION_ACTIVE) session_start();
    if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
    return $_SESSION['csrf'];
  }
}
if (!function_exists('verify_csrf')) {
  function verify_csrf($t){
    return (session_status()===PHP_SESSION_ACTIVE
            && !empty($_SESSION['csrf'])
            && hash_equals($_SESSION['csrf'], (string)$t));
  }
}

/* flash redirect */
function flash_redirect(string $msg='Saved', string $type='ok'){
  $qs = $_GET; unset($qs['add'], $qs['edit']);
  $qs[$type] = $msg;
  header('Location: ?'.http_build_query($qs)); exit;
}

ensure_schema($con, $TABLE, $STATUS_COL, $ORDER_COL);

/* ---------- lookups: profile types ---------- */
$PROFILE_TABLE = 'jos_app_profile_types'; // id, profile_name
$profiles = [];
$qr = $con->query("SELECT id, profile_name AS name FROM `$PROFILE_TABLE` ORDER BY profile_name ASC");
if ($qr) while ($r = $qr->fetch_assoc()) { $profiles[(int)$r['id']] = $r['name'] ?: ('#'.$r['id']); }

/* ---------- Access control helpers ----------
   Strategy:
   - If session admin has structured access data (common keys checked below), use it.
   - If no access info is present, default to ALLOW (preserve backward compatibility).
   - This function is conservative about unknown formats but looks for common patterns:
     - $_SESSION['admin_user']['permissions'] as list/assoc
     - $_SESSION['admin_user']['access'][<module>][<action>] = 1
     - $_SESSION['admin_user']['can_add_coupons'] style boolean flags
*/
function current_module_key(): string {
  // identifier for this module - used when checking structured access arrays
  return 'coupon_codes';
}
function has_access(string $action): bool {
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  $admin = $_SESSION['admin_user'] ?? null;
  if (!$admin) return true; // no admin info -> do not block (compat)

  // exact boolean flags like can_add_coupons, can_edit_coupons, etc.
  $flag = 'can_'.$action.'_'.current_module_key();
  if (isset($admin[$flag])) return (bool)$admin[$flag];

  // generic flags like can_add, can_edit (module-agnostic)
  if (isset($admin['can_'.$action])) return (bool)$admin['can_'.$action];

  // permissions array (list)
  if (!empty($admin['permissions']) && is_array($admin['permissions'])) {
    // permission strings might be like "coupon_codes.add" or "coupon_codes_add"
    $needleA = current_module_key().'.'.$action;
    $needleB = current_module_key().'_'.$action;
    if (in_array($needleA, $admin['permissions'], true) || in_array($needleB, $admin['permissions'], true)) {
      return true;
    }
    // also check for generic action
    if (in_array($action, $admin['permissions'], true)) return true;
  }

  // structured access array: access[module][action]
  if (!empty($admin['access']) && is_array($admin['access'])) {
    $mod = current_module_key();
    if (isset($admin['access'][$mod]) && is_array($admin['access'][$mod]) && isset($admin['access'][$mod][$action])) {
      return (bool)$admin['access'][$mod][$action];
    }
  }

  // role-based short-circuit: super / admin roles
  if (!empty($admin['role']) && in_array(strtolower($admin['role']), ['super','superadmin','administrator','admin'], true)) return true;

  // fallback: allow if no clear deny/list present (preserve existing behavior)
  return true;
}

/* ---------- load page title from jos_admin_menus (Menu Master) ----------
   Attempt to find a menu row that matches the current script name. Try several
   likely column names. If not found, use $DEFAULT_PAGE_TITLE.
*/
function load_title_from_menu(mysqli $con, string $fallback) : string {
  $script = basename($_SERVER['SCRIPT_NAME']);
  // attempt multiple likely column names for menu label and link
  $candidates = ['menu_name','title','label','name'];
  $link_cols   = ['menu_link','link','url','file'];
  foreach ($link_cols as $lcol) {
    $lcol_esc = preg_replace('/[^a-zA-Z0-9_]/','',$lcol);
    foreach ($candidates as $c) {
      $c_esc = preg_replace('/[^a-zA-Z0-9_]/','',$c);
      $sql = "SELECT `{$c_esc}` AS label FROM `jos_admin_menus`
              WHERE `{$lcol_esc}` LIKE ? LIMIT 1";
      if ($st = $con->prepare($sql)) {
        $like = '%'.$script.'%';
        $st->bind_param('s',$like);
        if ($st->execute()) {
          $res = $st->get_result();
          if ($row = $res->fetch_assoc()) {
            $st->close();
            return $row['label'] ?: $fallback;
          }
        }
        $st->close();
      }
    }
  }
  return $fallback;
}

$page_title = load_title_from_menu($con, $DEFAULT_PAGE_TITLE);

/* ---------- mode ---------- */
$mode = (isset($_GET['add']) || isset($_GET['edit'])) ? 'form' : 'list';

/* ---------- enforce view permission (UI-level) ----------
   If user cannot view, show a friendly message and stop further processing.
   NOTE: we still allow the script to be loaded so the message appears within the admin UI.
*/
$can_view   = has_access('view');
$can_add    = has_access('add');
$can_edit   = has_access('edit');
$can_delete = has_access('delete');

if (!$can_view && $mode === 'list') {
  ob_start();
  ?><link rel="stylesheet" href="/adminconsole/assets/ui.css">
  <div class="master-wrap">
    <div class="headbar"><h2><?=h($page_title)?></h2></div>
    <div class="card">
      <div class="alert err">You do not have permission to view this page.</div>
    </div>
  </div><?php
  echo ob_get_clean();
  exit;
}

/* ---------- POST handler ---------- */
$err = ''; $ok = '';
if ($_SERVER['REQUEST_METHOD']==='POST'){
  if (!verify_csrf($_POST['csrf'] ?? '')) { die('CSRF failed'); }

  if (isset($_POST['save'])){
    // check add/edit permission
    $id = (int)($_POST['id'] ?? 0);
    if ($id>0 && !$can_edit) { $err = 'You do not have permission to edit.'; }
    if ($id===0 && !$can_add) { $err = 'You do not have permission to add.'; }

    $coupon_code     = clean($_POST['coupon_code'] ?? '');
    $discount_type   = strtolower(clean($_POST['discount_type'] ?? 'percentage'));
    $discount_value  = (float)($_POST['discount_value'] ?? 0);
    $expiry_date     = clean($_POST['expiry_date'] ?? ''); // expect YYYY-MM-DD
    $profile_type_id = (int)($_POST['profile_type_id'] ?? 0);
    $max_usage_limit = (int)($_POST['max_usage_limit'] ?? 0);
    $status          = (int)($_POST['status'] ?? 1);

    // basic validations
    if (!$err && $coupon_code==='') $err = 'Coupon Code is required.';
    if (!$err && !in_array($discount_type, ['percentage','flat'], true)) $err = 'Invalid Discount Type.';
    if (!$err && $discount_value < 0) $err = 'Discount Value cannot be negative.';
    if (!$err && $expiry_date!=='' && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $expiry_date)) $err = 'Invalid Expiry Date (use YYYY-MM-DD).';
    if (!$err && $profile_type_id>0 && !isset($profiles[$profile_type_id])) $err = 'Invalid Profile Type.';

    // duplicate check on coupon_code
    if (!$err){
      $sql = "SELECT id FROM $TABLE WHERE LOWER($LABEL_COL)=LOWER(?) ".($id>0?"AND id<>$id":"")." LIMIT 1";
      $st  = $con->prepare($sql); $st->bind_param('s',$coupon_code);
      $st->execute(); $dup = $st->get_result()->fetch_assoc(); $st->close();
      if ($dup) $err = 'Duplicate Coupon Code.';
    }

    // insert / update
    $admin = $_SESSION['admin_user'] ?? [];
    $admin_id = (int)($admin['id'] ?? 0);

    if (!$err){
      if ($id>0){
        if (!$can_edit) { $err = 'You do not have permission to edit.'; }
        else {
          $sql = "UPDATE $TABLE
                  SET coupon_code=?, discount_type=?, discount_value=?, expiry_date=?, profile_type_id=?,
                      max_usage_limit=?, $STATUS_COL=?, updated_at=NOW()
                  WHERE id=?";
          $st = $con->prepare($sql);
          $st->bind_param('ssdsiiii',
            $coupon_code, $discount_type, $discount_value, $expiry_date, $profile_type_id,
            $max_usage_limit, $status, $id
          );
          if ($st->execute()) { $st->close(); flash_redirect('Updated successfully'); }
          $err = 'Update failed'; $st->close();
        }
      } else {
        if (!$can_add) { $err = 'You do not have permission to add.'; }
        else {
          $sql = "INSERT INTO $TABLE
                  (coupon_code, discount_type, discount_value, expiry_date, profile_type_id,
                   max_usage_limit, used_count, $STATUS_COL, created_by_userid, created_at, updated_at)
                  VALUES (?,?,?,?,?,?,?,?,?,NOW(),NOW())";
          $st = $con->prepare($sql);
          $used_count = 0;
          $st->bind_param('ssdsiiiii',
            $coupon_code, $discount_type, $discount_value, $expiry_date, $profile_type_id,
            $max_usage_limit, $used_count, $status, $admin_id
          );
          if ($st->execute()) { $st->close(); flash_redirect('Saved successfully'); }
          $err = 'Insert failed'; $st->close();
        }
      }
    }
  }

  if (isset($_POST['delete'])){
    if (!$can_delete) { $err = 'You do not have permission to delete.'; }
    else {
      $id = (int)($_POST['id'] ?? 0);
      $st = $con->prepare("DELETE FROM $TABLE WHERE id=?");
      $st->bind_param('i',$id);
      if ($st->execute()) { $st->close(); flash_redirect('Deleted successfully'); }
      $err = 'Delete failed'; $st->close();
    }
  }
}

/* ---------- edit row ---------- */
$edit = null;
if ($mode==='form' && isset($_GET['edit'])){
  $eid=(int)$_GET['edit'];
  $st=$con->prepare("SELECT * FROM $TABLE WHERE id=? LIMIT 1");
  $st->bind_param('i',$eid); $st->execute();
  $edit = $st->get_result()->fetch_assoc(); $st->close();
}

/* ---------- filters ---------- */
$q        = clean($_GET['q'] ?? '');
$sort     = clean($_GET['sort'] ?? 'order_asc');
$all      = isset($_GET['all']);
$limit    = $all ? 1000000 : 50;

/* ---------- list query ---------- */
$order_sql = "ORDER BY $ORDER_COL ASC";
if ($sort==='order_desc') $order_sql = "ORDER BY $ORDER_COL DESC";
if ($sort==='name_asc')   $order_sql = "ORDER BY $LABEL_COL ASC";
if ($sort==='name_desc')  $order_sql = "ORDER BY $LABEL_COL DESC";
if ($sort==='id_desc')    $order_sql = "ORDER BY id DESC";
if ($sort==='id_asc')     $order_sql = "ORDER BY id ASC";

$where = "WHERE 1";
if ($q!==''){ $qLike='%'.$con->real_escape_string($q).'%'; $where.=" AND $LABEL_COL LIKE '$qLike'"; }

$sql = "SELECT id, coupon_code, discount_type, discount_value, expiry_date,
               profile_type_id, max_usage_limit, used_count, $STATUS_COL AS status
        FROM $TABLE $where $order_sql LIMIT $limit";
$res = $con->query($sql);

/* ---------- view ---------- */
ob_start(); ?>
<link rel="stylesheet" href="/adminconsole/assets/ui.css">
<style>
  .toolbar { display:flex; justify-content:space-between; align-items:flex-end; gap:12px; }
  .toolbar .form { flex:1; }
  .grid-filters { display:grid; grid-template-columns: 1fr 160px 160px; gap:10px; }
  .inp { width:100%; }
  .alert.ok { background:#14532d; color:#d1fae5; padding:8px 12px; border-radius:6px; margin:10px 0; }
  .alert.err,.alert.warn { background:#7c2d12; color:#fee2e2; padding:8px 12px; border-radius:6px; margin:10px 0; }
  .small { font-size:12px; opacity:.8; }
  .mono { font-family: ui-monospace, Menlo, Consolas, monospace; }
</style>

<div class="master-wrap">
  <div class="headbar">
    <h2><?=h($page_title)?></h2>
  </div>

  <?php if ($mode==='list'): ?>
    <?php if(isset($_GET['ok'])): ?><div class="alert ok"><?=h($_GET['ok'])?></div><?php endif; ?>
    <?php if(isset($_GET['err'])): ?><div class="alert err"><?=h($_GET['err'])?></div><?php endif; ?>

    <div class="card">
      <div class="toolbar">
        <!-- LEFT: filters -->
        <form method="get" class="form">
          <div class="grid-filters">
            <input class="inp" type="text" name="q" value="<?=h($q)?>" placeholder="Search coupon code...">
            <select class="inp" name="sort">
              <option value="order_asc"  <?=($sort==='order_asc'?'selected':'')?>>Order Level ↑</option>
              <option value="order_desc" <?=($sort==='order_desc'?'selected':'')?>>Order Level ↓</option>
              <option value="name_asc"   <?=($sort==='name_asc'?'selected':'')?>>Code A–Z</option>
              <option value="name_desc"  <?=($sort==='name_desc'?'selected':'')?>>Code Z–A</option>
              <option value="id_desc"    <?=($sort==='id_desc'?'selected':'')?>>Newest first</option>
              <option value="id_asc"     <?=($sort==='id_asc'?'selected':'')?>>Oldest first</option>
            </select>
            <button class="btn gray" type="submit">Apply</button>
          </div>
        </form>

        <!-- RIGHT: actions -->
        <div class="toolbar" style="gap:8px;">
          <?php if (!isset($_GET['all'])): ?>
            <a class="btn secondary" href="<?=keep_params(['all'=>1])?>">View All</a>
          <?php else: ?>
            <a class="btn secondary" href="<?=keep_params(['all'=>null])?>">Last 50</a>
          <?php endif; ?>
          <?php if ($can_add): ?>
            <a class="btn success" href="<?=keep_params(['add'=>1])?>">+ Add New</a>
          <?php endif; ?>
        </div>
      </div>

      <div class="table-wrap">
        <table class="table">
          <thead style="position:sticky; top:0;">
            <tr>
              <th style="width:70px;">SR No</th>
              <th>Coupon Code</th>
              <th>Type/Value</th>
              <th>Expiry</th>
              <th>Profile Type</th>
              <th>Usage</th>
              <th>Status</th>
              <th style="width:160px;">Actions</th>
            </tr>
          </thead>
          <tbody>
          <?php
          if ($res && $res->num_rows){
            $sr=1;
            while($r=$res->fetch_assoc()){
              $ptname = $profiles[(int)$r['profile_type_id']] ?? ('#'.$r['profile_type_id']);
              $typeBadge = ($r['discount_type']==='flat' ? '₹' : '%');
              $valStr = ($r['discount_type']==='flat')
                        ? number_format((float)$r['discount_value'],2)
                        : rtrim(rtrim(number_format((float)$r['discount_value'],2,'.',''), '0'), '.');
              $exp = $r['expiry_date'] ? date('d-M-Y', strtotime($r['expiry_date'])) : '';
              echo '<tr>';
              echo '<td>'.($sr++).'</td>';
              echo '<td class="mono">'.h($r['coupon_code']).'</td>';
              echo '<td>'.$typeBadge.' '.$valStr.'</td>';
              echo '<td>'.$exp.'</td>';
              echo '<td>'.h($ptname).'</td>';
              echo '<td>'.(int)$r['used_count'].' / '.(int)$r['max_usage_limit'].'</td>';
              echo '<td><span class="badge '.((int)$r['status']?'success':'danger').'">'.((int)$r['status']?'Active':'Inactive').'</span></td>';
              echo '<td>';
              if ($can_edit) {
                echo '<a class="btn secondary" href="'.keep_params(['edit'=>$r['id']]).'">Edit</a> ';
              } else {
                echo '<button class="btn gray" disabled>View</button> ';
              }
              if ($can_delete) {
                echo '<form method="post" style="display:inline" onsubmit="return confirm(\'Delete?\');">';
                echo '<input type="hidden" name="csrf" value="'.h(csrf_token()).'">';
                echo '<input type="hidden" name="id" value="'.$r['id'].'">';
                echo '<button class="btn danger" name="delete" type="submit">Delete</button>';
                echo '</form>';
              } else {
                echo '<button class="btn gray" disabled>Delete</button>';
              }
              echo '</td>';
              echo '</tr>';
            }
          } else {
            echo '<tr><td colspan="8">No records.</td></tr>';
          }
          ?>
          </tbody>
        </table>
      </div>
    </div>

  <?php else: /* form */ 
    // default row
    $row = [
      'id'=>0,'coupon_code'=>'','discount_type'=>'percentage','discount_value'=>'',
      'expiry_date'=>'','profile_type_id'=>0,'max_usage_limit'=>0,'status'=>1,
    ];
    if ($edit){ $row = array_merge($row, $edit); }

    // Back to list: preserve query params (filters), but remove add/edit
    $LIST_URL = h($_SERVER['SCRIPT_NAME'] . keep_params(['add'=>null,'edit'=>null]));
  ?>
    <div class="card" style="max-width:980px">
      <div class="toolbar" style="justify-content:space-between">
        <h3 style="margin:0"><?= $row['id']? 'Edit Coupon':'Add Coupon' ?></h3>
        <a class="btn secondary" href="<?=$LIST_URL?>">Back to List</a>
      </div>

      <?php if($ok): ?><div class="alert ok"><?=h($ok)?></div><?php endif; ?>
      <?php if($err): ?><div class="alert err"><?=h($err)?></div><?php endif; ?>

      <form method="post" class="form">
        <input type="hidden" name="csrf" value="<?=h(csrf_token())?>">
        <?php if($row['id']): ?><input type="hidden" name="id" value="<?=$row['id']?>"><?php endif; ?>

        <div class="grid" style="grid-template-columns: repeat(auto-fit,minmax(220px,1fr)); gap:12px">
          <div class="inp">
            <label>Coupon Code *</label>
            <input class="inp" name="coupon_code" required maxlength="64" autocomplete="off"
                   value="<?=h($row['coupon_code'])?>" <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
          </div>

          <div class="inp">
            <label>Discount Type *</label>
            <?php $sel = strtolower($row['discount_type'] ?? 'percentage'); ?>
            <select class="inp" name="discount_type" required <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
              <option value="percentage" <?=($sel==='percentage'?'selected':'')?>>Percentage (%)</option>
              <option value="flat"       <?=($sel==='flat'?'selected':'')?>>Flat (₹)</option>
            </select>
          </div>

          <div class="inp">
            <label>Discount Value *</label>
            <input class="inp" type="number" step="0.01" min="0" name="discount_value" required
                   value="<?=h($row['discount_value'])?>" <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
            <small class="small">If Percentage, enter %; if Flat, enter rupee amount.</small>
          </div>

          <div class="inp">
            <label>Expiry Date *</label>
            <input class="inp" type="date" name="expiry_date" required
                   value="<?=h($row['expiry_date'] ? substr($row['expiry_date'],0,10) : '')?>" <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
          </div>

          <div class="inp">
            <label>Profile Type *</label>
            <select class="inp" name="profile_type_id" required <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
              <option value="">-- Select --</option>
              <?php
                $cur = (int)($row['profile_type_id'] ?? 0);
                foreach ($profiles as $pid=>$pname) {
                  $s = ($cur === $pid) ? 'selected' : '';
                  echo "<option value='".(int)$pid."' $s>".h($pname)." (#$pid)</option>";
                }
              ?>
            </select>
          </div>

          <div class="inp">
            <label>Max Usage Limit *</label>
            <input class="inp" type="number" min="0" name="max_usage_limit" required
                   value="<?=h((string)($row['max_usage_limit'] ?? 0))?>" <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
          </div>

          <div class="inp">
            <label>Status</label>
            <select class="inp" name="status" <?=(!$can_add && !$can_edit ? 'disabled' : '')?>>
              <option value="1" <?=((int)$row['status']===1?'selected':'')?>>Active</option>
              <option value="0" <?=((int)$row['status']===0?'selected':'')?>>Inactive</option>
            </select>
          </div>
        </div>

        <div class="toolbar" style="margin-top:12px">
          <?php if (($row['id'] && $can_edit) || (!$row['id'] && $can_add)): ?>
            <button class="btn success" name="save" type="submit">Save</button>
          <?php else: ?>
            <button class="btn gray" disabled>Save</button>
          <?php endif; ?>
        </div>
      </form>
    </div>
  <?php endif; ?>
</div>

<?php
echo ob_get_clean();
